import os
import requests
import json

# API ключ для CoreGPT
API_KEY = os.getenv("COREGPT_API_KEY", "96CC4-2B9A7-3959B-D8EB7")
API_URL = "https://core-gpt.com/v1/chat/completions"

# Промпт для деперсонализации
DEPERSONALIZE_PROMPT = (
    "Ты деперсонализатор. Твоя задача — заменить все персональные данные на вымышленные. \n"
    "Правила: \n"
    "- Заменяй ФИО на вымышленные, в стиле героев марвел. \n"
    "- Паспортные данные, адреса, номера телефонов, email и другие идентифицирующие данные должны быть заменены на случайные, но в том же формате. \n"
    "- Все данные, такие как адрес, должны иметь реалистичные, но вымышленные значения. \n"
    "- Не забывай сохранять формат текста и его структуру. \n"
    "- Например, дата рождения должна быть реальной по виду, но вымышленной по содержанию. \n"
    "- Ответ пиши на Русском языке без ошибок, как грамотный филолог. \n"
    "- Ответ должен содержать только деперсонализированный текст, без комментариев и вводных фраз. \n"
    "Входной текст (Деперсонализируй его): \n\n\{text}\n"
)

# Ограничение на размер текста
MAX_TEXT_LENGTH = 8000

# Функция для отправки запроса в CoreGPT
def get_ai_response(text):
    # Обрезаем текст, если его длина больше 8000 символов
    if len(text) > MAX_TEXT_LENGTH:
        text = text[:MAX_TEXT_LENGTH]

    prompt = DEPERSONALIZE_PROMPT.format(text=text)

    headers = {
        "Authorization": f"Bearer {API_KEY}",
        "Content-Type": "application/json",
    }

    payload = {
        "model": "CoreGPT",
        "messages": [{"role": "user", "content": prompt}],
        "stream": True,
    }

    try:
        response = requests.post(API_URL, headers=headers, data=json.dumps(payload), stream=True)
        response.raise_for_status()
        result = ""
        for line in response.iter_lines():
            if line:  # Пропускаем пустые строки
                decoded_line = line.decode("utf-8").strip()
                if decoded_line.startswith("data: "):
                    try:
                        chunk = json.loads(decoded_line[6:])
                        delta = chunk.get("choices", [{}])[0].get("delta", {}).get("content", "")
                        result += delta
                        # Выводим данные по мере их поступления
                        print(delta, end="", flush=True)  # Печатает символы по мере их поступления
                    except json.JSONDecodeError:
                        continue
        print()  # Завершаем строку после генерации
        return result.strip()
    except Exception as e:
        print(f"Ошибка при запросе к API: {e}")
        return None

# Функция для обработки файла
def process_file(input_path, output_path):
    try:
        with open(input_path, "r", encoding="utf-8") as f:
            text = f.read().strip()  # Убираем лишние пробелы и символы новой строки

        # Выводим исходный файл
        print("="*50)
        print(f"Исходный файл ({input_path}):")
        print("="*50)
        print(text)
        print("="*50)

        # Печать обработанного файла
        print(f"Деперсонализированный файл ({output_path}):")
        print("="*50, end="\n")
        depersonalized_text = get_ai_response(text)

        if depersonalized_text is not None:
            os.makedirs(os.path.dirname(output_path), exist_ok=True)
            with open(output_path, "w", encoding="utf-8") as f:
                f.write(depersonalized_text)
            print("="*50, end="\n\n")
        else:
            print(f"Не удалось обработать файл: {input_path}")
    except Exception as e:
        print(f"Ошибка при обработке файла {input_path}: {e}")

# Функция для обработки всех файлов в директории
def process_directory(input_dir, output_dir):
    for root, _, files in os.walk(input_dir):
        for file in files:
            if file.endswith(".txt"):
                input_path = os.path.join(root, file)
                relative_path = os.path.relpath(input_path, input_dir)
                output_path = os.path.join(output_dir, relative_path)
                process_file(input_path, output_path)

# Основная функция
def main():
    input_dir = os.path.join(os.getcwd(), "input")
    output_dir = os.path.join(os.getcwd(), "output")

    if not os.path.exists(input_dir):
        print(f"Папка 'input' не найдена: {input_dir}")
        return

    if not os.path.exists(output_dir):
        os.makedirs(output_dir)

    process_directory(input_dir, output_dir)
    print("\nВсе файлы обработаны.")

if __name__ == "__main__":
    main()
